/*
 * Decompiled with CFR 0.152.
 */
package zeenea.connector.commons.cache;

import java.io.Closeable;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Iterator;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import zeenea.connector.commons.cache.CacheData;
import zeenea.connector.commons.cache.CacheEntry;
import zeenea.connector.commons.cache.CachePageLoader;
import zeenea.connector.commons.cache.CacheReload;
import zeenea.connector.commons.cache.HeaderPage;
import zeenea.connector.commons.cache.InterruptedCacheException;
import zeenea.connector.commons.cache.InvalidCacheConfigException;
import zeenea.connector.commons.cache.InvalidCacheException;
import zeenea.connector.commons.cache.PageLoader;
import zeenea.connector.commons.cache.TableDesc;
import zeenea.connector.commons.cache.TimeoutCacheException;
import zeenea.connector.commons.cache.ZeeCacheConfig;
import zeenea.connector.commons.cache.ZeeCacheLoader;
import zeenea.connector.commons.cache.ZeeCacheTable;
import zeenea.connector.commons.cache.ZeeCacheWriter;

public final class ZeeCache
implements Closeable {
    private static final int VERSION = 2;
    private static final Duration RELOAD_TIMEOUT = Duration.ofHours(6L);
    private static final Logger log = LoggerFactory.getLogger(ZeeCache.class);
    private static final ConcurrentHashMap<String, CacheUsage> cacheCache = new ConcurrentHashMap();
    @Nullable
    private final String name;
    @NotNull
    private final CachePageLoader pageLoader;
    @Nullable
    private ZeeCacheTable rootTable;

    private ZeeCache(@Nullable String name, @NotNull CachePageLoader pageLoader) {
        this.name = name;
        this.pageLoader = pageLoader;
    }

    public static ZeeCache of(int applicationVersion, @NotNull ZeeCacheConfig config, @Nullable CacheReload shouldReload, @Nullable ZeeCacheLoader cacheLoader) {
        Path path = config.path();
        if (path != null) {
            return ZeeCache.file(applicationVersion, path, config.memPageCount(), config.directAllocation(), config.timeToLive(), shouldReload, cacheLoader);
        }
        return ZeeCache.memory(applicationVersion, cacheLoader);
    }

    public static ZeeCache of(int applicationVersion, @NotNull ZeeCacheConfig config, @Nullable ZeeCacheLoader cacheLoader) {
        return ZeeCache.of(applicationVersion, config, CacheReload.noReload(), cacheLoader);
    }

    private static ZeeCache file(int applicationVersion, Path path, int memPageCount, boolean directAllocation, Duration timeToLive, CacheReload forceReload, ZeeCacheLoader cacheLoader) {
        Objects.requireNonNull(path);
        if (timeToLive != null && cacheLoader == null) {
            throw new InvalidCacheConfigException("A cache loader should be defined when a time to live is set.");
        }
        String cacheName = path.toString();
        CacheUsage usage = cacheCache.computeIfAbsent(cacheName, __ -> new CacheUsage());
        usage.lock.lock();
        try {
            long timeout = RELOAD_TIMEOUT.toNanos();
            while (usage.used()) {
                if (timeout <= 0L) {
                    throw new TimeoutCacheException("Timeout while waiting to reload the cache");
                }
                CacheReload reload = ZeeCache.shouldReload(usage.cache.pageLoader, applicationVersion, timeToLive, forceReload, cacheLoader);
                if (!reload.needed()) {
                    log.debug("zeecache_file_reuse_existing_cache cache='{}'", (Object)cacheName);
                    usage.use();
                    ZeeCache zeeCache = usage.cache;
                    return zeeCache;
                }
                log.debug("zeecache_file_should_reload_existing_cache cache='{}' cause='{}'", (Object)cacheName, (Object)reload.cause());
                timeout = usage.waitReleased(timeout);
            }
            log.debug("zeecache_file_create_new_cache cache='{}'", (Object)cacheName);
            CachePageLoader pageLoader = CachePageLoader.of(PageLoader.open(path, memPageCount, directAllocation));
            try {
                CacheReload reload = ZeeCache.shouldReload(pageLoader, applicationVersion, timeToLive, forceReload, cacheLoader);
                ZeeCache.loadCache(reload, pageLoader, applicationVersion, cacheLoader);
                usage.cache = new ZeeCache(cacheName, pageLoader);
                usage.use();
                ZeeCache zeeCache = usage.cache;
                return zeeCache;
            }
            catch (Throwable e) {
                try {
                    pageLoader.close();
                    throw e;
                }
                catch (InterruptedException e2) {
                    throw new InterruptedCacheException("Interrupted while waiting to reload the cache", e2);
                }
            }
        }
        finally {
            usage.lock.unlock();
        }
    }

    public static ZeeCache file(int applicationVersion, @NotNull Path path) {
        return ZeeCache.file(applicationVersion, path, 2048, true, null, CacheReload.noReload(), null);
    }

    public static ZeeCache memory(int applicationVersion, ZeeCacheLoader cacheLoader) {
        if (cacheLoader == null) {
            throw new InvalidCacheConfigException("A cache loader should be defined for memory cache.");
        }
        CachePageLoader pageLoader = CachePageLoader.of(PageLoader.memory());
        ZeeCache.loadCache(CacheReload.reload("memory cache should be reload"), pageLoader, applicationVersion, cacheLoader);
        return new ZeeCache(null, pageLoader);
    }

    private static CacheReload shouldReload(CachePageLoader pageLoader, int appVersion, @Nullable Duration timeToLive, @Nullable CacheReload shouldReload, @Nullable ZeeCacheLoader cacheLoader) {
        if (shouldReload != null && shouldReload.needed()) {
            return shouldReload;
        }
        if (pageLoader.hasPage(0)) {
            HeaderPage headerPage = pageLoader.headerPage();
            if (!headerPage.isCacheHeader()) {
                pageLoader.close();
                throw new InvalidCacheException("Invalid cache file header");
            }
            int fileVersion = headerPage.version();
            if (fileVersion != 2) {
                return CacheReload.reload("cache version changed");
            }
            int fileApplicationVersion = headerPage.applicationVersion();
            if (fileApplicationVersion != appVersion && appVersion != 0) {
                return CacheReload.reload("application version changed");
            }
            Instant fileDate = headerPage.cacheDate();
            if (timeToLive != null && Instant.now().isAfter(fileDate.plus(timeToLive))) {
                return CacheReload.reload("cache expired");
            }
            return CacheReload.noReload();
        }
        if (cacheLoader != null) {
            return CacheReload.reload("Cache file doesn't exist or is empty");
        }
        pageLoader.close();
        throw new InvalidCacheException("File doesn't exist and no cacheLoader was provided");
    }

    private static void loadCache(CacheReload reload, CachePageLoader pageLoader, int applicationVersion, ZeeCacheLoader cacheLoader) {
        if (reload.needed()) {
            log.info("zeecache_load_cache_start cause='{}'", (Object)reload.cause());
            pageLoader.startLoading();
            HeaderPage headerPage = pageLoader.headerPage();
            headerPage.init(2, applicationVersion, Instant.now());
            ZeeCacheWriter cacheWriter = new ZeeCacheWriter(pageLoader, headerPage);
            cacheLoader.load(cacheWriter);
            cacheWriter.close();
            pageLoader.stopLoading();
            log.info("zeecache_load_cache_success");
        }
    }

    @Nullable
    public ZeeCacheTable getTable(String name) {
        CacheData value = this.rootTable().get(name);
        if (value == null) {
            return null;
        }
        TableDesc tableDesc = TableDesc.read(value);
        return ZeeCacheTable.of(name, this.pageLoader, tableDesc);
    }

    @NotNull
    public Iterator<ZeeCacheTable> tableIterator() {
        return new TableIterator(this.rootTable().iterator());
    }

    private ZeeCacheTable rootTable() {
        if (this.rootTable == null) {
            HeaderPage headerPage = this.pageLoader.headerPage();
            TableDesc tableDesc = headerPage.rootTable();
            this.rootTable = ZeeCacheTable.of("", this.pageLoader, tableDesc);
        }
        return this.rootTable;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void close() {
        CacheUsage usage;
        CacheUsage cacheUsage = usage = this.name != null ? cacheCache.get(this.name) : null;
        if (usage != null) {
            usage.lock.lock();
            try {
                if (usage.cache == this && !usage.release()) return;
                this.pageLoader.close();
                return;
            }
            finally {
                usage.lock.unlock();
            }
        } else {
            this.pageLoader.close();
        }
    }

    private static final class CacheUsage {
        private static final Logger log = LoggerFactory.getLogger(CacheUsage.class);
        final Lock lock = new ReentrantLock();
        final Condition released = this.lock.newCondition();
        int using;
        ZeeCache cache;

        private CacheUsage() {
        }

        void use() {
            ++this.using;
            log.debug("zeecache_usage_use cache='{}' using='{}'", (Object)this.cache.name, (Object)this.using);
        }

        boolean used() {
            return this.using > 0;
        }

        long waitReleased(long nanos) throws InterruptedException {
            String cacheName = this.cache.name;
            log.debug("zeecache_wait_release_start cache='{}' delay='{}'", (Object)cacheName, (Object)nanos);
            long remaining = this.released.awaitNanos(nanos);
            log.debug("zeecache_wait_release_end cache='{}' remaining='{}'", (Object)cacheName, (Object)remaining);
            return remaining;
        }

        boolean release() {
            log.debug("zeecache_release cache='{}' using='{}'", (Object)this.cache.name, (Object)this.using);
            --this.using;
            if (!this.used()) {
                log.debug("zeecache_release_do_release cache='{}' ", (Object)this.cache.name);
                this.cache = null;
                this.released.signalAll();
                return true;
            }
            log.debug("zeecache_release_used cache='{}' using='{}'", (Object)this.cache.name, (Object)this.using);
            return false;
        }
    }

    private final class TableIterator
    implements Iterator<ZeeCacheTable> {
        private final Iterator<CacheEntry> iterator;

        TableIterator(Iterator<CacheEntry> iterator) {
            this.iterator = iterator;
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public ZeeCacheTable next() {
            CacheEntry next = this.iterator.next();
            TableDesc tableDesc = TableDesc.read(next.value());
            return ZeeCacheTable.of(next.key().stringValue(), ZeeCache.this.pageLoader, tableDesc);
        }
    }
}

