/*
 * Decompiled with CFR 0.152.
 */
package zeenea.connector.commons.cache;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.Closeable;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.jetbrains.annotations.NotNull;
import zeenea.connector.commons.cache.CacheTimeoutException;
import zeenea.connector.commons.cache.FileCacheChannelHandler;
import zeenea.connector.commons.cache.InterruptedCacheException;
import zeenea.connector.commons.cache.InvalidCacheStateException;
import zeenea.connector.commons.cache.InvalidPageException;
import zeenea.connector.commons.cache.Page;
import zeenea.connector.commons.cache.PageLoader;
import zeenea.connector.commons.cache.PageLoaderState;
import zeenea.connector.commons.cache.PageState;
import zeenea.connector.commons.cache.ReadOnlyPageException;
import zeenea.connector.commons.cache.ResourceExhaustedException;

class FilePageLoader
implements PageLoader,
Closeable {
    public static final int DEFAULT_MEM_PAGE_COUNT = 2048;
    private static final Duration PAGE_TIMEOUT = Duration.ofMinutes(10L);
    private int pageCount;
    private PageLoaderState state;
    private final FileCacheChannelHandler channelHandler;
    private final Page[] memPages;
    private final Map<Integer, Page> memPagesByNumber;
    private Page firstFree;
    private Page lastFree;
    private final Lock lock = new ReentrantLock();
    private final Condition morePages = this.lock.newCondition();

    FilePageLoader(Path path, int memPageCount, boolean directAllocation) {
        this.state = PageLoaderState.READ;
        this.channelHandler = new FileCacheChannelHandler(this, path);
        this.memPages = new Page[memPageCount];
        this.memPagesByNumber = new HashMap<Integer, Page>(2 * memPageCount);
        this.initPageCount();
        this.createPages(memPageCount, directAllocation);
        this.createFreePageList(memPageCount);
    }

    private void initPageCount() {
        switch (this.state) {
            case READ: {
                this.pageCount = this.channelHandler.loadPageCount();
                break;
            }
            case LOAD: {
                this.pageCount = 0;
                break;
            }
            case CLOSED: {
                throw FilePageLoader.cacheIsClosed();
            }
        }
    }

    private void createPages(int memPageCount, boolean directAllocation) {
        ByteBuffer buffer = directAllocation ? ByteBuffer.allocateDirect(memPageCount * 4096) : ByteBuffer.allocate(memPageCount * 4096);
        for (int i = 0; i < memPageCount; ++i) {
            Page page;
            buffer.limit((i + 1) * 4096);
            buffer.position(i * 4096);
            ByteBuffer pageBuffer = buffer.slice();
            this.memPages[i] = page = Page.of(this, pageBuffer);
        }
    }

    private void createFreePageList(int memPageCount) {
        Page previousPage;
        this.firstFree = previousPage = this.memPages[0];
        for (int i = 1; i < memPageCount; ++i) {
            Page page = this.memPages[i];
            previousPage.setNextFree(page);
            page.setPreviousFree(previousPage);
            previousPage = page;
        }
        previousPage.setNextFree(null);
        this.lastFree = previousPage;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void startLoading() {
        this.lock.lock();
        try {
            switch (this.state) {
                case READ: {
                    Page[] pageArray = this.memPages;
                    int n = pageArray.length;
                    int n2 = 0;
                    while (true) {
                        if (n2 >= n) {
                            this.memPagesByNumber.clear();
                            this.channelHandler.closeChannel();
                            this.state = PageLoaderState.LOAD;
                            this.initPageCount();
                            return;
                        }
                        Page page = pageArray[n2];
                        page.reset();
                        ++n2;
                    }
                }
                case LOAD: {
                    return;
                }
                case CLOSED: {
                    throw FilePageLoader.cacheIsClosed();
                }
            }
            return;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void stopLoading() {
        this.lock.lock();
        try {
            switch (this.state) {
                case READ: {
                    return;
                }
                case LOAD: {
                    this.saveAll();
                    this.channelHandler.closeChannel();
                    this.state = PageLoaderState.READ;
                    this.initPageCount();
                    return;
                }
                case CLOSED: {
                    throw FilePageLoader.cacheIsClosed();
                }
            }
            return;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void close() {
        this.lock.lock();
        try {
            switch (this.state) {
                case LOAD: {
                    try {
                        this.saveAll();
                        return;
                    }
                    finally {
                        this.state = PageLoaderState.CLOSED;
                        this.channelHandler.closeChannel();
                    }
                }
                case READ: {
                    this.state = PageLoaderState.CLOSED;
                    this.channelHandler.closeChannel();
                    return;
                }
                case CLOSED: {
                    return;
                }
            }
            return;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean readOnly() {
        this.lock.lock();
        try {
            if (this.state == PageLoaderState.CLOSED) {
                throw FilePageLoader.cacheIsClosed();
            }
            boolean bl = this.state == PageLoaderState.READ;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean hasPage(int pageNumber) {
        this.lock.lock();
        try {
            boolean bl = pageNumber < this.pageCount;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Page newPage() {
        this.lock.lock();
        try {
            int pageNumber;
            if (this.state != PageLoaderState.LOAD) {
                throw new ReadOnlyPageException("Cannot create new pages in readonly mode");
            }
            if ((pageNumber = this.pageCount++) < 0) {
                throw new ResourceExhaustedException("Can't have more that 2147483647 pages");
            }
            Page page = this.nextFreePage(pageNumber);
            page.setState(PageState.LOADED);
            this.memPagesByNumber.put(pageNumber, page);
            Page page2 = page;
            return page2;
        }
        catch (InterruptedException e) {
            throw new InterruptedCacheException("Interrupted while taking a free page", e);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Page page(int pageNumber) {
        this.lock.lock();
        try {
            Page page = this.memPagesByNumber.get(pageNumber);
            if (page == null) {
                if (pageNumber >= this.pageCount) {
                    throw new InvalidPageException("The page " + pageNumber + " doesn't exist.");
                }
                page = this.nextFreePage(pageNumber);
                this.channelHandler.read(page);
                page.setState(PageState.LOADED);
                this.memPagesByNumber.put(pageNumber, page);
            } else {
                if (!page.isBorrowed()) {
                    if (page == this.lastFree && page == this.firstFree) {
                        this.lastFree = null;
                        this.firstFree = null;
                    } else if (page == this.lastFree) {
                        this.lastFree = page.previousFree();
                        this.lastFree.setNextFree(null);
                    } else if (page == this.firstFree) {
                        this.firstFree = page.nextFree();
                        this.firstFree.setPreviousFree(null);
                    } else {
                        page.previousFree().setNextFree(page.nextFree());
                        page.nextFree().setPreviousFree(page.previousFree());
                    }
                    page.setNextFree(null);
                    page.setPreviousFree(null);
                }
                page.borrow();
            }
            Page page2 = page;
            return page2;
        }
        catch (InterruptedException e) {
            throw new InterruptedCacheException("Interrupted while waiting for a free page", e);
        }
        finally {
            this.lock.unlock();
        }
    }

    @SuppressFBWarnings(value={"SF_SWITCH_FALLTHROUGH"})
    private Page nextFreePage(int pageNumber) throws InterruptedException {
        long pageTimeout = PAGE_TIMEOUT.toNanos();
        while (this.firstFree == null) {
            if (pageTimeout <= 0L) {
                throw new CacheTimeoutException("Timeout when waiting for page " + pageNumber + " for 10 minutes");
            }
            pageTimeout = this.morePages.awaitNanos(pageTimeout);
        }
        Page page = this.firstFree;
        this.firstFree = page.nextFree();
        if (this.firstFree != null) {
            this.firstFree.setPreviousFree(null);
        } else {
            this.lastFree = null;
        }
        switch (page.state()) {
            case MODIFIED: {
                this.channelHandler.save(page);
            }
            case LOADED: {
                this.memPagesByNumber.remove(page.number());
                page.clear();
            }
            case FREE: {
                page.setNumber(pageNumber);
            }
        }
        page.setNextFree(null);
        page.setPreviousFree(null);
        page.borrow();
        return page;
    }

    @Override
    public void release(Page page) {
        this.lock.lock();
        try {
            if (page.release()) {
                if (this.lastFree != null) {
                    page.setPreviousFree(this.lastFree);
                    this.lastFree.setNextFree(page);
                } else {
                    this.firstFree = page;
                }
                this.lastFree = page;
                this.morePages.signal();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void saveAll() {
        if (this.state == PageLoaderState.LOAD) {
            for (Page page : this.memPages) {
                if (page.state() != PageState.MODIFIED) continue;
                this.channelHandler.save(page);
                page.setState(PageState.LOADED);
            }
        }
    }

    @NotNull
    static InvalidCacheStateException cacheIsClosed() {
        return new InvalidCacheStateException("File Cache is closed");
    }
}

